//
//  CMS_Strings.swift
//  Composr Mobile SDK
//
//  Created by Aaswini on 06/08/14.
//  Copyright (c) 2014 Aaswini. All rights reserved.
//
/*
 
 CMS_Strings
 
 string strip_tags(string str) - remove all HTML tags from the input string
 string html_entity_decode(string str) - convert common HTML entities to standard characters (e.g. "&amp;" becomes "&", "&ldquo;" becomes the unicode equivalent of that HTML entity)
 string float_format(float number, int decimalPoints, bool onlyIncludeNeededDecimalPoints) - formats a number nicely (e.g. with commas and decimal points)
 int strpos(string searchIn, string searchFor) - see the PHP manual
 string str_replace(string search, string replace, string searchIn) - see the PHP manual
 string substr(string searchIn, int offset, int length) - see the PHP manual
 string trim(string str) - see the PHP manual

 */

import UIKit
import Foundation

class CMS_Strings: NSObject {
    
    /**
     *  Remove html tags from string
     *
     *  @param str source string
     *
     *  @return String after removing all html tags
     */
    class func strip_tags(str: String) -> String {
        var r: NSRange
        var s = str
        while (r = s.rangeOfString("<[^>]+>", options: .RegularExpressionSearch)).location != NSNotFound {
            s = s.stringByReplacingCharactersInRange(r, withString: "")
        }
        return s
    }
    
    /**
     *  Decode all html encoded values
     *
     *  @param str source string
     *
     *  @return String after decoding html encoded values
     */
    class func html_entity_decode(str: String) -> String {
        return str.stringByDecodingHTMLEntities()
    }
    
    /**
     *  Format a floating point number
     *
     *  @param number                         source number
     *  @param decimalPoints                  Number of required decimal digits
     *  @param onlyIncludeNeededDecimalPoints YES - include decimals, NO - remove all decimal places
     *
     *  @return returns formatted number as string
     */
    class func float_format(number: Double, decimalPoints: Int, onlyIncludeNeededDecimalPoints: Bool) -> String {
        var formatter = NSNumberFormatter()
        formatter.numberStyle = NSNumberFormatterDecimalStyle
        formatter.usesSignificantDigits = false
        formatter.usesGroupingSeparator = true
        formatter.groupingSeparator = ","
        formatter.decimalSeparator! = "."
        formatter.maximumFractionDigits = 25
        if onlyIncludeNeededDecimalPoints {
            formatter.maximumFractionDigits = decimalPoints
        }
        return formatter.stringFromNumber(Int(number))!
    }
    
    /**
     *  Returns position of a substring in string
     *
     *  @param searchIn  source string
     *  @param searchFor substring to search for
     *
     *  @return starting position of substring in the source string. -1 of not found.
     */
    class func strpos(searchIn: String, searchFor: String) -> Int {
        var range = (searchIn as NSString).rangeOfString(searchFor)
        if range.location == NSNotFound {
            return -1
        }
        return range.location
    }
    
    /**
     *  Replace occurances of substring with a given string
     *
     *  @param search   substring
     *  @param replace  replace string
     *  @param searchIn source string
     *
     *  @return replaced string
     */
    class func str_replace(search: String, replace: String, searchIn: String) -> String {
        return searchIn.stringByReplacingOccurrencesOfString(search, withString: replace)
    }
    
    /**
     *  Returns a substring from string
     *
     *  @param searchIn source string
     *  @param offset   starting point of search
     *  @param length   length of substring
     *
     *  @return Returns substring from source string from the offset position to the length given
     */
    class func substr(searchIn: String, offset: Int, length: Int) -> String {
        return searchIn.substringWithRange(NSMakeRange(offset, length))
    }
    
    /**
     *  Trim whitespaces at the begenning and ending of a string
     *
     *  @param str source string
     *
     *  @return Returns trimmed string
     */
    class func trim(str: String) -> String {
        return str.stringByTrimmingCharactersInSet<NSObject>(NSCharacterSet.whitespaceAndNewlineCharacterSet<NSObject>())
    }
    
    /**
     *  Returns length of a string
     *
     *  @param str Input string
     *
     *  @return length of the input string
     */
    class func strlen(str: String) -> Int {
        return Int(str.length)
    }
    
    /**
     *  Replaces arguments given into the string with placeholders
     *
     *  @param format    source string with place holders. eg: I like %@ and %@ .
     *  @param arguments values to be replaced into the string eq : [ "cats" , "dogs" ]
     *
     *  @return returned formatted string
     */
    class func stringWithFormat(format: String, array arguments: [AnyObject]) -> String {
        if arguments.count > 10 {
            NSException(name: NSRangeException, reason: "Maximum of 10 arguments allowed", userInfo: ["collection": arguments])
        }
        var a = arguments + ["X", "X", "X", "X", "X", "X", "X", "X", "X", "X"]
        return String(format: format, a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9])
    }
}